import { rules } from "./rules.js";

/* =========================
   UTILITIES
========================= */

function calculateEntropy(str) {
  const map = {};
  for (const c of str) map[c] = (map[c] || 0) + 1;

  let entropy = 0;
  for (const c in map) {
    const p = map[c] / str.length;
    entropy -= p * Math.log2(p);
  }
  return entropy;
}

function extractRootDomain(hostname) {
  const parts = hostname.split(".");
  if (parts.length <= 2) return hostname;
  return parts.slice(-2).join(".");
}

function normalizeHomoglyphs(str) {
  return str
    .replace(/0/g, "o")
    .replace(/1/g, "l")
    .replace(/3/g, "e")
    .replace(/5/g, "s")
    .replace(/7/g, "t");
}

function editDistance(a, b) {
  const dp = Array.from({ length: a.length + 1 }, () =>
    Array(b.length + 1).fill(0)
  );

  for (let i = 0; i <= a.length; i++) dp[i][0] = i;
  for (let j = 0; j <= b.length; j++) dp[0][j] = j;

  for (let i = 1; i <= a.length; i++) {
    for (let j = 1; j <= b.length; j++) {
      dp[i][j] =
        a[i - 1] === b[j - 1]
          ? dp[i - 1][j - 1]
          : Math.min(
              dp[i - 1][j] + 1,
              dp[i][j - 1] + 1,
              dp[i - 1][j - 1] + 1
            );
    }
  }
  return dp[a.length][b.length];
}

/* =========================
   BRAND DATABASE
========================= */

const BRAND_DOMAINS = {
  paypal: "paypal.com",
  google: "google.com",
  apple: "apple.com",
  icloud: "icloud.com",
  microsoft: "microsoft.com",
  amazon: "amazon.com",
  netflix: "netflix.com",
  facebook: "facebook.com",
  instagram: "instagram.com",
  x: "x.com",
  linkedin: "linkedin.com",
  github: "github.com",
  cloudflare: "cloudflare.com",
  binance: "binance.com",
  coinbase: "coinbase.com",
  epicgames: "epicgames.com",
  robinhood: "robinhood.com"
};

/* =========================
   MAIN ANALYSIS
========================= */

export function analyzeURL(input) {
  let hostname;

  try {
    const parsed = new URL(input.startsWith("http") ? input : "https://" + input);
    hostname = parsed.hostname.toLowerCase();
  } catch {
    return {
      verdict: "WARNING",
      score: 30,
      reasons: ["Invalid or malformed URL"],
      explanation: "The URL format is invalid."
    };
  }

  const rootDomain = extractRootDomain(hostname);
  const rawLabel = rootDomain.split(".")[0];
  const normalizedLabel = normalizeHomoglyphs(rawLabel);

  /* =========================
     OFFICIAL DOMAIN WHITELIST
  ========================= */

  for (const brand in BRAND_DOMAINS) {
    const official = BRAND_DOMAINS[brand];

    if (
      rootDomain === official ||
      hostname.endsWith("." + official)
    ) {
      return {
        verdict: "SAFE",
        score: 0,
        reasons: [],
        explanation: "Official trusted domain."
      };
    }
  }

  /* =========================
     BRAND IMPERSONATION
  ========================= */

  for (const brand in BRAND_DOMAINS) {
    if (normalizedLabel === brand) {
      return {
        verdict: "DANGEROUS",
        score: 95,
        reasons: [
          `Brand impersonation detected: "${brand}"`
        ],
        explanation:
          "This domain directly impersonates a trusted brand."
      };
    }
  }

  /* =========================
     TYPOSQUATTING
  ========================= */

  for (const brand in BRAND_DOMAINS) {
    const distance = editDistance(normalizedLabel, brand);

    if (
      distance > 0 &&
      distance <= 2 &&
      normalizedLabel.length >= brand.length - 1
    ) {
      return {
        verdict: "DANGEROUS",
        score: 85,
        reasons: [
          `Typosquatting detected: "${rawLabel}" mimics "${brand}"`
        ],
        explanation:
          "This domain closely imitates a known brand."
      };
    }
  }

  /* =========================
     HEURISTICS
  ========================= */

  let score = 0;
  const reasons = [];

  if (input.length > 80) {
    score += 8;
    reasons.push("Unusually long URL");
  }

  if (/^\d+\.\d+\.\d+\.\d+$/.test(hostname)) {
    score += 25;
    reasons.push("Uses IP address instead of domain");
  }

  rules.suspiciousKeywords.forEach(word => {
    if (hostname.includes(word)) {
      score += 6;
      reasons.push(`Suspicious keyword: "${word}"`);
    }
  });

  rules.dangerousTlds.forEach(tld => {
    if (hostname.endsWith(tld)) {
      score += 15;
      reasons.push(`Risky TLD: ${tld}`);
    }
  });

  const subdomains = hostname.split(".").length - 2;
  if (subdomains > rules.maxSubdomains) {
    score += 10;
    reasons.push("Too many subdomains");
  }

  const entropy = calculateEntropy(rawLabel);
  if (entropy > rules.entropyThreshold) {
    score += 12;
    reasons.push(`High entropy domain (${entropy.toFixed(2)})`);
  }

  /* =========================
     FINAL VERDICT
  ========================= */

  if (score >= 12) {
    return {
      verdict: "WARNING",
      score,
      reasons,
      explanation: "Suspicious patterns detected."
    };
  }

  return {
    verdict: "SAFE",
    score,
    reasons,
    explanation: "No significant risk detected."
  };
}
